/**
 * @customElement
 * @polymer
 */
class PlanechaseApp extends Polymer.Element {
  static get is() { return 'planechase-app'; }
  
  static get properties() {
    return {
      cards: Array,
      errors: {
        type: Array,
        value: () => [],
      },
      singleDeckOption: Boolean,
      numPlayers: {
        type: Number,
        observer: 'numPlayersChanged_',
      },
    };
  }
  
  ready() {
    super.ready();
    const single = JSON.parse(localStorage.getItem('singleDeckOption'));
    this.singleDeckOption = single;
    this.numPlayers = JSON.parse(localStorage.getItem('numPlayers'));
  }
  
  numPlayersChanged_(newValue) {
    localStorage.setItem('numPlayers', JSON.stringify(newValue));
    this.checkErrors_();
  }
  
  hasErrors() {
    return !!this.errors.length;
  }
  
  openSettings(e) {
    const srcButton = e.path.find((el) => el.localName === 'paper-icon-button');
    this.$.settings.positionTarget = srcButton;
    this.$.settings.open();
  }
  
  resetDeck() {
    this.$.card.setCard();
    this.$.deck.cards = this.enabledCards();
    for (let i = this.$.deck.cards.length - 1; i > 0; i--) {
      const n = Math.floor(Math.random() * (i + 1));
      const tmp = this.$.deck.cards[i];
      this.$.deck.cards[i] = this.$.deck.cards[n];
      this.$.deck.cards[n] = tmp;
    }
  }
  
  rollDie() {
    if (this.timer) {
      return;
    }
    this.$.die.style.display = 'inline-block';
    setTimeout(() => {
      this.$.die.rollDie();
      this.timer = setTimeout(() => {
        this.$.die.style.display = 'none';
        this.timer = null;
      }, 2500);
    }, 50);
  }
  
  nameToId(name) {
    return name.toLowerCase().replace(/[ ']/g, ($0) => {
      if ($0 === ' ') {
        return '-';
      } else {
        return '';
      }
    });
  }
  
  cardDataLoaded(e) {
    this.cards = e.detail.response.cards;
    this.cards.forEach((card) => {
      const img = new Image();
      img.src = card.src;
      card.img = img;
    });
    if (!localStorage.getItem('cards')) {
      localStorage.setItem('cards',
          JSON.stringify(this.cards.map((el) => ({name: el.name}))));
    }
    const available = JSON.parse(localStorage.getItem('cards'));
    setTimeout(() => {
      available.forEach((card) => {
        const checkbox = this.shadowRoot
            .querySelector(`#${this.nameToId(card.name)}`);
        checkbox.checked = !!card.disabled;
      });
      
      let cards = [...this.$.pc1.nextElementSibling
          .querySelectorAll('paper-checkbox')];
      this.$.pc1.checked = cards.every((checkbox) => checkbox.checked);
      
      cards = [...this.$.pc2.nextElementSibling
          .querySelectorAll('paper-checkbox')];
      this.$.pc2.checked = cards.every((checkbox) => checkbox.checked);
      
      this.resetDeck();
      this.checkErrors_();
    });
  }
  
  cardsFrom(release) {
    return (card) => card.set === release;
  }
  
  enabledCards() {
    const cards = JSON.parse(localStorage.getItem('cards'));
    return cards.filter((card) => !card.disabled).map((card) => card.name);
  }
  
  updateAvailableCard(e) {
    const checkbox = e.path.find((el) => el.localName === 'paper-checkbox');
    const card = checkbox.textContent;
    const checked = checkbox.checked;
    const available = JSON.parse(localStorage.getItem('cards'));
    available.find((el) => el.name === card).disabled = checked;
    localStorage.setItem('cards', JSON.stringify(available));
    
    const checkboxGroup = [...checkbox.parentNode.parentNode.children]
        .filter((el) => el.localName === 'li')
        .map((li) => li.children[0]);
    checkbox.parentNode.parentNode.previousElementSibling.checked =
        checkboxGroup.every((checkbox) => checkbox.checked);
    
    this.checkErrors_();
  }
  
  toggleSetAvailable(e) {
    const checkbox = e.path.find((el) => el.localName === 'paper-checkbox');
    const cards = [...checkbox.nextElementSibling
        .querySelectorAll('paper-checkbox')];
    const available = JSON.parse(localStorage.getItem('cards'));
    cards.forEach((card) => {
      card.checked = checkbox.checked;
      available.find((el) => el.name === card.textContent).disabled =
          card.checked;
    });
    localStorage.setItem('cards', JSON.stringify(available));
    
    this.checkErrors_();
  }
  
  toggleSinglePlanarDeck(e) {
    const checked = e.target.checked;
    localStorage.setItem('singleDeckOption', JSON.stringify(checked));
    this.checkErrors_();
  }
  
  checkErrors_() {
    this.errors = [];
    if (!this.cards) return;
    
    const available = JSON.parse(localStorage.getItem('cards'))
        .filter((card) => !card.disabled);
    if (available.find((card) => card.name === 'Spatial Merging')) {
      this.push('errors', 'Spatial Merging is not implemented.');
    }
    if (available.find((card) => card.name === 'Stairs to Infinity')) {
      this.push('errors', 'Stairs to Infinity is not implemented.');
    }
    
    const numPlanes = available.reduce((memo, curr) => {
      const data = this.cards.find((card) => card.name === curr.name);
      const isPhenomenon = data.isPhenomenon;
      return memo + (isPhenomenon ? 0 : 1);
    }, 0);
    if (available.find((card) => card.name === 'Interplanar Tunnel')
        && numPlanes < 5) {
      this.push('errors',
          'Interplanar Tunnel requires at least five plane cards.');
    }
    
    const numPhenomena = available.reduce((memo, curr) => {
      const data = this.cards.find((card) => card.name === curr.name);
      const isPhenomenon = data.isPhenomenon;
      return memo+  (isPhenomenon ? 1 : 0);
    }, 0);
    if (this.singleDeckOption) {
      let requiredAmt = Math.min(40, 10 * this.numPlayers);
      if (available.length < requiredAmt) {
        this.push('errors', `Planar deck must contain at least ${requiredAmt}
            cards.`)
      }
      requiredAmt = 2 * this.numPlayers;
      if (numPhenomena > requiredAmt) {
        this.push('errors', `Planar deck cannot contain more than ${requiredAmt}
            phenomena.`);
      }
    } else {
      if (available.length < 10) {
        this.push('errors', 'Planar deck must contain at least 10 cards.');
      }
      if (numPhenomena > 2) {
        this.push('errors',
            'Planar deck cannot contain more than two phenomena.');
      }
    }
  }
  
  handlePlaneswalk(e) {
    if (this.$.card.hasCard) {
      this.$.deck.cards.push(this.$.card.card.name);
    }
    this.$.card.setCard(e.detail);
  }
  
  forcePlaneswalk() {
    this.$.deck.tappedDeck();
  }
  
  encounterChaoticAether() {
    this.$.die.chaoticAether = true;
  }
  
  endChaoticAether() {
    this.$.die.chaoticAether = false;
  }
  
  handleRolledPlaneswalk() {
    this.$.card.dispatchEvent(new CustomEvent('rolled-planeswalk', {
      bubbles: true,
      composed: true,
    }));
  }
  
  handleRolledChaos() {
    this.$.card.dispatchEvent(new CustomEvent('rolled-chaos', {
      bubbles: true,
      composed: true,
    }));
  }
  
  handlePoolsOfBecoming() {
    this.$.deck.handlePoolsOfBecoming();
  }
  
  handleInterplanarTunnel() {
    this.$.deck.handleInterplanarTunnel();
  }
}

window.customElements.define(PlanechaseApp.is, PlanechaseApp);